<?php
header('Content-Type: application/json; charset=utf-8');
require_once 'dbconnection.php'; // usa $con (mysqli)

/**
 * Helpers JSON
 */
function json_ok($data = array(), $code = 200) {
    if (function_exists('http_response_code')) http_response_code($code);
    echo json_encode($data);
    exit;
}
function json_error($message, $code = 400, $extra = array()) {
    if (function_exists('http_response_code')) http_response_code($code);
    echo json_encode(array_merge(array(
        'error'   => true,
        'message' => $message,
    ), $extra));
    exit;
}
function read_json_body() {
    $raw = file_get_contents('php://input');
    if (!$raw) return array();
    $data = json_decode($raw, true);
    if (json_last_error() !== JSON_ERROR_NONE) return array();
    return $data;
}

$action = isset($_GET['action']) ? $_GET['action'] : null;
if (!$action) json_error('Acción no especificada', 400);

switch ($action) {
    case 'init_conversation':
        init_conversation($con);
        break;

    case 'get_messages':
        get_messages_resident($con);
        break;

    case 'send_message':
        send_message_resident($con);
        break;

    case 'send_file':
        send_file_resident($con);
        break;

    case 'unread_user_count':
        unread_user_count($con);
        break;

    case 'conversation_info':
        conversation_info($con);
        break;

    case 'admin_status':
        admin_status($con);
        break;

    default:
        json_error('Acción no reconocida', 400);
}

/**
 * ======================
 *   init_conversation
 * ======================
 *
 * Busca si ya hay una conversación ABIERTA para ese condominio+residente.
 * - Si hay abierta → devuelve conversation_id y last_status = 'open'
 * - Si no hay → NO crea nada, solo devuelve el último status (si hubo chats antes)
 */
function init_conversation($con)
{
    $data       = read_json_body();
    $condoId    = isset($data['condominio_id']) ? intval($data['condominio_id']) : 0;
    $residentId = isset($data['residente_id'])  ? intval($data['residente_id'])  : 0;

    if ($condoId <= 0 || $residentId <= 0) {
        json_error('condominio_id y residente_id son requeridos', 400);
    }

    // 1) Buscar conversación ABIERTA (segura con varias variantes de status)
    $sqlOpen = "
        SELECT id, status
        FROM chat_conversations
        WHERE condominio_id = {$condoId}
          AND residente_id  = {$residentId}
          AND (
                status = 'open'
             OR status = 'Open'
             OR status = 'abierto'
             OR status = 'Abierto'
          )
        ORDER BY updated_at DESC, id DESC
        LIMIT 1
    ";
    $resOpen = mysqli_query($con, $sqlOpen);
    if (!$resOpen) {
        json_error('Error buscando conversación abierta', 500, array('detail' => mysqli_error($con)));
    }

    if ($openRow = mysqli_fetch_assoc($resOpen)) {
        json_ok(array(
            'success'         => true,
            'conversation_id' => (int)$openRow['id'],
            'last_status'     => 'open',
            'created'         => false
        ));
    }

    // 2) No hay conversación abierta → buscamos la última (puede estar cerrada o no existir)
    $sqlLast = "
        SELECT id, status
        FROM chat_conversations
        WHERE condominio_id = {$condoId}
          AND residente_id  = {$residentId}
        ORDER BY updated_at DESC, id DESC
        LIMIT 1
    ";
    $resLast = mysqli_query($con, $sqlLast);
    if (!$resLast) {
        json_error('Error buscando última conversación', 500, array('detail' => mysqli_error($con)));
    }

    $lastRow   = mysqli_fetch_assoc($resLast);
    $lastStatus = $lastRow ? $lastRow['status'] : null;

    json_ok(array(
        'success'         => true,
        'conversation_id' => null,               // NO se crea todavía
        'last_status'     => $lastStatus,
        'created'         => false
    ));
}

/**
 * ======================
 *   get_messages_resident
 * ======================
 *
 * Devuelve TODO el historial de mensajes de una conversación
 * (admin + residente + adjuntos).
 */
function get_messages_resident($con)
{
    $chatId = isset($_GET['chat_id']) ? intval($_GET['chat_id']) : 0;
    if ($chatId <= 0) json_error('chat_id es requerido', 400);

    $sql = "
        SELECT id, author_type, author_name, text, file_url, file_name, created_at
        FROM chat_messages
        WHERE conversation_id = {$chatId}
        ORDER BY created_at ASC, id ASC
    ";
    $res = mysqli_query($con, $sql);
    if (!$res) {
        json_error('Error consultando mensajes', 500, array('detail' => mysqli_error($con)));
    }

    $messages = array();
    while ($row = mysqli_fetch_assoc($res)) {
        $messages[] = array(
            'id'          => (int)$row['id'],
            'author_type' => $row['author_type'],
            'author_name' => $row['author_name'],
            'text'        => $row['text'],
            'file_url'    => $row['file_url'],
            'file_name'   => $row['file_name'],
            'created_at'  => $row['created_at'],
        );
    }

    // El residente ya vio el chat → limpiar has_unread_user
    $sqlRead = "UPDATE chat_conversations SET has_unread_user = 0 WHERE id = {$chatId}";
    mysqli_query($con, $sqlRead);

    json_ok($messages);
}

/**
 * ======================
 *   send_message_resident
 * ======================
 */
function send_message_resident($con)
{
    $data   = read_json_body();
    $chatId = isset($data['chat_id']) ? intval($data['chat_id']) : 0;
    $text   = isset($data['message']) ? trim($data['message']) : '';
    $condoId    = isset($data['condominio_id']) ? intval($data['condominio_id']) : 0;
    $residentId = isset($data['residente_id'])  ? intval($data['residente_id'])  : 0;

    if ($text === '') {
        json_error('message es requerido', 400);
    }

    // Si NO hay chat_id, creamos conversación AHORA (primer mensaje)
    if ($chatId <= 0) {
        if ($condoId <= 0 || $residentId <= 0) {
            json_error('condominio_id y residente_id son requeridos para crear chat', 400);
        }

        $sqlInsConv = "
            INSERT INTO chat_conversations
                (condominio_id, residente_id, status, has_unread_admin, has_unread_user, updated_at)
            VALUES
                ({$condoId}, {$residentId}, 'open', 1, 0, NOW())
        ";
        if (!mysqli_query($con, $sqlInsConv)) {
            json_error('Error creando conversación', 500, array('detail' => mysqli_error($con)));
        }

        $chatId = mysqli_insert_id($con);
    } else {
        // Verificar que exista y no esté cerrada
        $sqlConv = "SELECT * FROM chat_conversations WHERE id = {$chatId} LIMIT 1";
        $resConv = mysqli_query($con, $sqlConv);
        if (!$resConv) json_error('Error consultando conversación', 500, array('detail' => mysqli_error($con)));
        $conv = mysqli_fetch_assoc($resConv);
        if (!$conv) json_error('Conversación no encontrada', 404);
        if (strtolower(trim($conv['status'])) === 'closed' || strtolower(trim($conv['status'])) === 'cerrado') {
            json_error('La conversación está cerrada', 400);
        }
    }

    $authorType = 'user';
    $authorName = 'Residente';

    $msgEsc  = mysqli_real_escape_string($con, $text);
    $nameEsc = mysqli_real_escape_string($con, $authorName);
    $typeEsc = mysqli_real_escape_string($con, $authorType);

    $sqlInsMsg = "
        INSERT INTO chat_messages (conversation_id, author_type, author_name, text, created_at)
        VALUES ({$chatId}, '{$typeEsc}', '{$nameEsc}', '{$msgEsc}', NOW())
    ";
    if (!mysqli_query($con, $sqlInsMsg)) {
        json_error('Error insertando mensaje', 500, array('detail' => mysqli_error($con)));
    }

    // Flags: admin tiene algo nuevo
    $sqlUpd = "
        UPDATE chat_conversations
        SET has_unread_admin = 1,
            has_unread_user  = 0,
            updated_at = NOW()
        WHERE id = {$chatId}
    ";
    mysqli_query($con, $sqlUpd);

    json_ok(array(
        'success'         => true,
        'conversation_id' => (int)$chatId
    ));
}

/**
 * ======================
 *   send_file_resident
 * ======================
 * Recibe un archivo de FormData (name="file") + chat_id opcional.
 */
function send_file_resident($con)
{
    $chatId      = isset($_POST['chat_id']) ? intval($_POST['chat_id']) : 0;
    $condoId     = isset($_POST['condominio_id']) ? intval($_POST['condominio_id']) : 0;
    $residentId  = isset($_POST['residente_id'])  ? intval($_POST['residente_id'])  : 0;

    if (!isset($_FILES['file']) || $_FILES['file']['error'] != UPLOAD_ERR_OK) {
        json_error('Archivo no recibido o con error', 400);
    }

    // Si no hay chat_id, creamos conversación
    if ($chatId <= 0) {
        if ($condoId <= 0 || $residentId <= 0) {
            json_error('condominio_id y residente_id son requeridos para enviar archivo', 400);
        }

        $sqlInsConv = "
            INSERT INTO chat_conversations
                (condominio_id, residente_id, status, has_unread_admin, has_unread_user, updated_at)
            VALUES
                ({$condoId}, {$residentId}, 'open', 1, 0, NOW())
        ";
        if (!mysqli_query($con, $sqlInsConv)) {
            json_error('Error creando conversación para archivo', 500, array('detail' => mysqli_error($con)));
        }

        $chatId = mysqli_insert_id($con);
    } else {
        // Verificar conversación
        $sqlConv = "SELECT * FROM chat_conversations WHERE id = {$chatId} LIMIT 1";
        $resConv = mysqli_query($con, $sqlConv);
        if (!$resConv) json_error('Error consultando conversación', 500, array('detail' => mysqli_error($con)));
        $conv = mysqli_fetch_assoc($resConv);
        if (!$conv) json_error('Conversación no encontrada', 404);
        if (strtolower(trim($conv['status'])) === 'closed' || strtolower(trim($conv['status'])) === 'cerrado') {
            json_error('La conversación está cerrada', 400);
        }
    }

    // Carpeta de subida
    $uploadDir = __DIR__ . '/uploads/chat/';
    if (!is_dir($uploadDir)) {
        @mkdir($uploadDir, 0777, true);
    }

    $originalName = $_FILES['file']['name'];
    $ext = pathinfo($originalName, PATHINFO_EXTENSION);
    $safeName = preg_replace('/[^a-zA-Z0-9_\-\.]/', '_', pathinfo($originalName, PATHINFO_FILENAME));
    $finalName = $safeName . '_' . time() . ($ext ? '.' . $ext : '');
    $path = $uploadDir . $finalName;

    if (!move_uploaded_file($_FILES['file']['tmp_name'], $path)) {
        json_error('No se pudo mover el archivo', 500);
    }

    // URL pública relativa (ajusta si usas otra ruta)
    $publicUrl = 'uploads/chat/' . $finalName;

    $authorType = 'user';
    $authorName = 'Residente';

    $nameEsc  = mysqli_real_escape_string($con, $authorName);
    $typeEsc  = mysqli_real_escape_string($con, $authorType);
    $fileUrl  = mysqli_real_escape_string($con, $publicUrl);
    $fileName = mysqli_real_escape_string($con, $originalName);

    $sqlInsMsg = "
        INSERT INTO chat_messages (conversation_id, author_type, author_name, text, file_url, file_name, created_at)
        VALUES ({$chatId}, '{$typeEsc}', '{$nameEsc}', '', '{$fileUrl}', '{$fileName}', NOW())
    ";
    if (!mysqli_query($con, $sqlInsMsg)) {
        json_error('Error insertando mensaje con archivo', 500, array('detail' => mysqli_error($con)));
    }

    $sqlUpd = "
        UPDATE chat_conversations
        SET has_unread_admin = 1,
            has_unread_user  = 0,
            updated_at = NOW()
        WHERE id = {$chatId}
    ";
    mysqli_query($con, $sqlUpd);

    json_ok(array(
        'success'         => true,
        'conversation_id' => (int)$chatId,
        'file_url'        => $publicUrl
    ));
}

/**
 * ======================
 *   unread_user_count
 * ======================
 */
function unread_user_count($con)
{
    $condoId    = isset($_GET['condominio_id']) ? intval($_GET['condominio_id']) : 0;
    $residentId = isset($_GET['residente_id'])  ? intval($_GET['residente_id'])  : 0;

    if ($condoId <= 0 || $residentId <= 0) {
        json_error('condominio_id y residente_id son requeridos', 400);
    }

    $sql = "
        SELECT COUNT(*) AS total
        FROM chat_conversations
        WHERE condominio_id = {$condoId}
          AND residente_id  = {$residentId}
          AND status = 'open'
          AND has_unread_user = 1
    ";
    $res = mysqli_query($con, $sql);
    if (!$res) {
        json_error('Error contando no leídos usuario', 500, array('detail' => mysqli_error($con)));
    }

    $row = mysqli_fetch_assoc($res);
    json_ok(array('total' => (int)($row['total'] ?? 0)));
}

/**
 * ======================
 *   conversation_info
 * ======================
 */
function conversation_info($con)
{
    $chatId = isset($_GET['chat_id']) ? intval($_GET['chat_id']) : 0;
    if ($chatId <= 0) json_error('chat_id es requerido', 400);

    $sql = "
        SELECT id, status, has_unread_user, has_unread_admin
        FROM chat_conversations
        WHERE id = {$chatId}
        LIMIT 1
    ";
    $res = mysqli_query($con, $sql);
    if (!$res) {
        json_error('Error consultando conversación', 500, array('detail' => mysqli_error($con)));
    }

    $row = mysqli_fetch_assoc($res);
    if (!$row) json_error('Conversación no encontrada', 404);

    json_ok($row);
}

/**
 * ======================
 *   admin_status
 * ======================
 *
 * Mira tb_users.last_activity para ver si hay admins online en los últimos X segundos.
 */
function admin_status($con)
{
    $windowSec = 60; // 1 minuto para pruebas
    $minTs = time() - $windowSec;

    $sql = "
        SELECT COUNT(*) AS total
        FROM tb_users
        WHERE groupname IN ('Administrador', 'Sub-Administrador', 'Super-Administrador')
          AND last_activity >= {$minTs}
    ";

    $res = mysqli_query($con, $sql);
    if (!$res) {
        json_error('Error consultando admins online', 500, array('detail' => mysqli_error($con)));
    }

    $row = mysqli_fetch_assoc($res);
    $count = (int)($row['total'] ?? 0);

    json_ok(array(
        'online'       => $count > 0,
        'online_count' => $count,
        'window_sec'   => $windowSec
    ));
}
